import axios from 'axios'
import Qs from 'qs'
import store from '@/store'
import router from '@/router'
import config from '@/config'
import { Message } from 'iview'
import { getToken } from '@/libs/util'
const baseUrl = process.env.NODE_ENV === 'development' ? config.baseUrl.dev : config.baseUrl.pro

const _baseConfig = {
  baseURL: baseUrl,
  headers: {},
  params: {},
  paramsSerializer: function (params) {
    return Qs.stringify(params)
  },
  validateStatus: status => {
    // 处理响应状态码
    switch (status) {
      case 401:
        router.push({ name: 'login' })
        return false
      case 500:
        // console.log(500);
        return false
    }
    return status >= 200 && status < 300 // 默认的, 为false时进入'响应拦截器'catch
  },
  transformResponse: [
    function (data) {
      // 对 data 进行任意转换处理
      // console.log(data);
      return data
    }
  ],
  timeout: 3 * 60 * 1000,
  responseType: 'json'
}

const _http = axios.create(_baseConfig)

// 请求拦截器
_http.interceptors.request.use(
  function (config) {
    config.setLoading && store.commit('setIsLoading', true)
    // 可以在这里修改请求头
    config.headers.Authorization = getToken()
    return config
  },
  function (error) {
    // 对请求错误做些什么
    return Promise.reject(error)
  }
)

// 响应拦截器
_http.interceptors.response.use(
  function (response) {
    console.log(response)
    // 在这里统一处理自定义状态码(如果有)
    // console.log(response.headers.pragma);
    store.commit('setIsLoading', false)
    let result = response.data
    if (result.code === -1) {
      Message.error(result.message)
    }
    if (typeof result === 'string') {
      result = JSON.parse(result)
    }
    return result
  },
  function (error) {
    if (!navigator.onLine) {
      console.log('无法连接网络')
    }
    store.commit('setIsLoading', false)
    return Promise.reject(error)
  }
)
const _HttpService = {
  post: (url, data, config) => {
    const _selfConfig = {
      headers: {
        'Content-Type': 'application/x-www-form-urlencoded;charset=utf-8'
      },
      setLoading: true
    }

    config = Object.assign({}, _selfConfig, config)
    return _http.post(url, Qs.stringify(data), config)
  },
  postObj: (url, data, config) => {
    const _selfConfig = {
      headers: {
        'Content-Type': 'application/json'
      },
      setLoading: true
    }

    config = Object.assign({}, _selfConfig, config)
    return _http.post(url, JSON.stringify(data), config)
  },
  postParams: (url, data, config) => {
    const _selfConfig = {
      params: data,
      setLoading: true
    }

    config = Object.assign({}, _selfConfig, config)
    return _http.post(url, null, config)
  },
  get: (url, data, config) => {
    const _selfConfig = {
      params: data,
      setLoading: true
    }

    config = Object.assign({}, _selfConfig, config)
    return _http.get(url, config)
  },
  delete: (url, data, config) => {
    const _selfConfig = {
      headers: {
        'Content-Type': 'application/json'
      },
      data: JSON.stringify(data),
      setLoading: true
    }

    config = Object.assign({}, _selfConfig, config)
    return _http.delete(url, config)
  },
  patch: (url, data, config) => {
    const _selfConfig = {
      headers: {
        'Content-Type': 'application/json'
      },
      setLoading: true
    }

    config = Object.assign({}, _selfConfig, config)
    return _http.patch(url, JSON.stringify(data), config)
  }
}

export const $http = _HttpService
export const $BASEURL = baseUrl
